import httpx
from . import console
import rich.progress
from pathlib import Path
import boto3

def get_signed_url(object_name: str) -> str:
    s3_client = boto3.client('s3')
    response = s3_client.generate_presigned_url(
        'get_object',
        Params={
            'Bucket': 'spacerake',
            'Key': object_name
        },
        ExpiresIn=(2 * 60 * 60) # two hours
    )

    return response


def download_dataset(dataset_name: str, output: Path):

    if Path(output).exists():
        console.log(f'[yellow]path {output} exists, skipping download...')
        return

    asset_url = get_signed_url(dataset_name)
    
    with httpx.stream('GET', asset_url) as response, open(output, 'wb') as out_file:
        total = int(response.headers['Content-Length'])

        with rich.progress.Progress(
                '[progress.percentage]{task.percentage:>3.0f}%',
                rich.progress.BarColumn(bar_width=None),
                rich.progress.DownloadColumn(),
                rich.progress.TransferSpeedColumn(),
                console=console
            ) as progress:
            download_task = progress.add_task('Download', total=total)
            for chunk in response.iter_bytes():
                out_file.write(chunk)
                progress.update(download_task, completed=response.num_bytes_downloaded)

