import httpx
from . import console
import rich.progress
from pathlib import Path

def get_signed_url(object_name: str) -> str:
    r = httpx.get(
        'https://sign.tiles.demo.internal.spacerake.net/api/v0/presign',
        params={
            'key': object_name
        }
    )
    r = r.json()

    if not 'ok' in r or not r['ok']:
        console.print(r)
        raise RuntimeError(f'couldn\'t sign url for key {object_name}')

    return r['signed_url']

def download_dataset(dataset_name: str, output: Path):

    if Path(output).exists():
        console.log(f'[yellow]path {output} exists, skipping download...')
        return

    asset_url = get_signed_url(dataset_name)
    
    with httpx.stream('GET', asset_url) as response, open(output, 'wb') as out_file:
        total = int(response.headers['Content-Length'])

        with rich.progress.Progress(
                '[progress.percentage]{task.percentage:>3.0f}%',
                rich.progress.BarColumn(bar_width=None),
                rich.progress.DownloadColumn(),
                rich.progress.TransferSpeedColumn(),
                console=console
            ) as progress:
            download_task = progress.add_task('Download', total=total)
            for chunk in response.iter_bytes():
                out_file.write(chunk)
                progress.update(download_task, completed=response.num_bytes_downloaded)

