from datetime import datetime
from enum import Enum
from typing import Any, List, Optional, Union, Literal

from pydantic import BaseModel
from geojson_pydantic import Polygon

from .base import Image, VendorNameEnum

# --- SEARCH FILTER METHODS ---

class Filter(BaseModel):
    pass


class TimeFilter(Filter):
    filter_type: Literal['time'] = 'time'

    start_time: datetime
    end_time: datetime


class ResolutionFilter(Filter):
    filter_type: Literal['resolution'] = 'resolution'

    minimum_resolution: Optional[float] = None
    maximum_resolution: Optional[float] = None


class ImageryTypeEnum(str, Enum):
    aerial = 'aerial'
    derivative = 'derivative'
    optical = 'optical'
    sar = 'sar'


class ImageryTypeFilter(Filter):
    filter_type: Literal['imagery_type'] = 'imagery_type'

    imagery_type: List[ImageryTypeEnum]


class CloudCoverageFilter(Filter):
    filter_type: Literal['cloud_coverage'] = 'cloud_coverage'

    cloud_cover: float


class OffNadirFilter(Filter):
    filter_type: Literal['off_nadir'] = 'off_nadir'

    off_nadir: int


class AOICoverageFilter(Filter):
    filter_type: Literal['aoi_coverage'] = 'aoi_coverage'

    min_aoi_coverage: float


ALL_FILTER_TYPES = [
    TimeFilter,
    ImageryTypeFilter,
    ResolutionFilter,
    CloudCoverageFilter,
    OffNadirFilter,
    AOICoverageFilter
]

# have to duplicate above, spreads are not allowed in types
AnyFilter = Union[
    TimeFilter,
    ImageryTypeFilter,
    ResolutionFilter,
    CloudCoverageFilter,
    OffNadirFilter,
    AOICoverageFilter
]

# --- SEARCH OBJS ---

class SearchRequest(BaseModel):
    polygon: Polygon
    filters: List[AnyFilter]
    vendors: Optional[List[VendorNameEnum]] = None


class SearchError(BaseModel):
    message: str


class Search(BaseModel):
    search_id: str
    user_id: str

    polygon: Polygon
    time: datetime

    filters: Optional[Any] = None
    vendors: Optional[List[VendorNameEnum]] = None


class SearchResponse(BaseModel):
    search: Search
    results: List[Image]
