from datetime import datetime
from enum import Enum
from typing import Any, Generic, Optional, TypeVar, List

from pydantic import BaseModel
from pydantic.functional_validators import field_validator
from geojson_pydantic import Polygon

# --- BASIC GEOMETRIES ---

# Polygon is actually exported from geojson_pydantic automatically, so
# we'll just leave it like that. From your code, you can just do:
# from sr_models.base import Polygon

class BoundingBox(BaseModel):
    north: float
    south: float
    east:  float
    west:  float

# --- SUPPORTED VENDORS ---

class VendorNameEnum(str, Enum):
    arlula   = 'arlula'
    up42     = 'up42'
    landsat  = 'landsat'
    sentinel = 'sentinel'
    goes     = 'goes'


class Provider(BaseModel):
    provider_id: Optional[str] = None
    name: str
    description: str


class Platform(BaseModel):
    platform_id: Optional[str] = None
    provider_id: str

    name: str
    description: str

    
class Image(BaseModel):
    image_id: Optional[str] = None
    search_id: Optional[str] = None
    vendor_id: str

    platform_id: str
    supplier_id: str

    scene_id: str

    time: datetime
    polygon: Polygon
    aoi_coverage: Optional[float] = None

    thumbnail: Optional[str] = None

    resolution: Optional[float] = None
    cloud_cover: Optional[float] = None
    off_nadir:   Optional[float] = None

    json_blob:   Optional[str] = None


class SavedLocationCreationRequest(BaseModel):
    location_footprint: Polygon
    location_json: Any = None
    location_osm: Optional[str] = None

    name: str
    description: str


class SavedLocation(SavedLocationCreationRequest):
    owner: str
    location_id: str


class Asset(BaseModel):
    asset_id: Optional[str] = None
    
    image_id: Optional[str] = None
    order_id: Optional[str] = None

    asset_name: str
    asset_size: int
    asset_mime: str
    asset_link: str

    asset_title: str
    asset_filename: str

    asset_expires: datetime


# --- ORDERS ---

class OrderStatusEnum(str, Enum):
    created    = 'created'
    waiting    = 'waiting'
    processing = 'processing'
    complete   = 'complete'
    failed     = 'failed'

class Order(BaseModel):
    order_id: Optional[str] = None
    image_id: Optional[str] = None
    owner: Optional[str] = None

    scene_id: Optional[str] = None

    purchase_date: datetime

    order_status: OrderStatusEnum
    price: int
    json_blob: Optional[str] = None


class OrderWithDetails(Order):
    image: Image
    assets: List[Asset]


class OrderCreationRequest(BaseModel):
    image_id: str


class APIError(BaseModel):
    message: str


DataT = TypeVar('DataT')
class APIResponse(BaseModel, Generic[DataT]):
    data: Optional[DataT]
    error: Optional[str]

    # do we really... need this?
    # TODO[pydantic]: We couldn't refactor the `validator`, please replace it by `field_validator` manually.
    # Check https://docs.pydantic.dev/dev-v2/migration/#changes-to-validators for more information.
    # @field_validator('error')
    # def check_consistency(cls, v, values):
    #     if v is not None and values['data'] is not None:
    #         raise ValueError('must not provide both data and error')
    #     if v is None and not 'data' in values:
    #         raise ValueError('must provide data or error')
    #     return v
