from datetime import datetime
from enum import Enum
from typing import Any, Generic, Optional, TypeVar, List

from pydantic import BaseModel, validator
from pydantic.generics import GenericModel

# temporary, eventually this will be locked down better
Polygon = Any

# --- SUPPORTED VENDORS ---

class VendorNameEnum(str, Enum):
    arlula   = 'arlula'
    up42     = 'up42'
    landsat  = 'landsat'
    sentinel = 'sentinel'


class Provider(BaseModel):
    provider_id: Optional[str]
    name: str
    description: str


class Platform(BaseModel):
    platform_id: Optional[str]
    provider_id: str

    name: str
    description: str

    
class Image(BaseModel):
    image_id: Optional[str]
    search_id: Optional[str]
    vendor_id: str

    platform_id: str
    supplier_id: str

    scene_id: str

    time: datetime
    polygon: Polygon

    thumbnail: Optional[str]

    resolution: Optional[float]
    cloud_cover: Optional[float]
    off_nadir:   Optional[float]

    json_blob:   Optional[str]


class SavedLocationCreationRequest(BaseModel):
    location_footprint: Polygon
    location_json: Any
    location_osm: Optional[str]

    name: str
    description: str


class SavedLocation(SavedLocationCreationRequest):
    owner: str
    location_id: str


class Asset(BaseModel):
    asset_id: Optional[str]
    
    image_id: Optional[str]
    order_id: Optional[str]

    asset_name: str
    asset_size: int
    asset_mime: str
    asset_link: str

    asset_title: str
    asset_filename: str

    asset_expires: datetime


# --- ORDERS ---

class OrderStatusEnum(str, Enum):
    created    = 'created'
    waiting    = 'waiting'
    processing = 'processing'
    complete   = 'complete'
    failed     = 'failed'

class Order(BaseModel):
    order_id: Optional[str]
    image_id: Optional[str]
    owner: Optional[str]

    scene_id: Optional[str]

    purchase_date: datetime

    order_status: OrderStatusEnum
    price: int
    json_blob: Optional[str]


class OrderWithDetails(Order):
    image: Image
    assets: List[Asset]


class OrderCreationRequest(BaseModel):
    image_id: str


class APIError(BaseModel):
    message: str


DataT = TypeVar('DataT')
class APIResponse(GenericModel, Generic[DataT]):
    data: Optional[DataT]
    error: Optional[str]

    @validator('error', always=True)
    def check_consistency(cls, v, values):
        if v is not None and values['data'] is not None:
            raise ValueError('must not provide both data and error')
        if v is None and values.get('data') is None:
            raise ValueError('must provide data or error')
        return v
