import asyncio
import functools
import inspect
import os
from typing import Any, List, Optional

import httpx
from sr_models.base import (
    APIResponse,
    Asset,
    Image,
    Order,
    OrderCreationRequest,
    OrderWithDetails,
    SavedLocation,
    SavedLocationCreationRequest,
)
from sr_models.user import UserProfile
from sr_models.lineage import LineageReport
from sr_models.search import Search, SearchRequest, SearchResponse


# this would be cool but mangles the static type inference in Pyright and Mypy
def optionally_synchronous(func):
    '''
    Wrap a function to block on async class methods if a field is set on the
    object (here, _methods_sync).
    '''

    @functools.wraps(func)
    def wrapper(*args, **kwargs):
        if inspect.iscoroutinefunction(func) and args[0]._methods_sync:
            coro = func(*args, **kwargs)
            return asyncio.get_event_loop().run_until_complete(coro)

        return func(*args, **kwargs)

    return wrapper


class SpacerakeSDKSync:
    def __init__(self, core):
        self.core = core

    def __getattr__(self, __name: str) -> Any:
        func = getattr(self.core, __name)

        @functools.wraps(func)
        def wrapper(*args, **kwargs):
            # print('wrapped coroutine!')
            coro = func(*args, **kwargs)
            if inspect.isawaitable(coro):
                return asyncio.get_event_loop().run_until_complete(coro)

            return coro

        return wrapper


class SpacerakeAPIException(Exception):
    pass


class SpacerakeSDK:
    api_url: str
    api_key: str
    api_secret: str
    
    def __init__(self, api_url: Optional[str], api_key: str, api_secret: str) -> None:
        '''
        Initialize the API connector.

        :param api_url: the API url to use. If none is provided it defaults to api.spacerake.net
        :type api_url: Optional[str]
        :param api_key: API key to use
        :type api_key: str
        :param api_secret: API secret to use
        :type api_secret: str
        '''
        self.api_url       = api_url if api_url is not None else 'api.spacerake.net'
        self.api_key       = api_key
        self.api_secret    = api_secret


    @staticmethod
    def from_env():
        '''
        Initialize the API connector from environment variables.

        Pulls from:
            - SPACERAKE_API_URL
            - SPACERAKE_API_KEY
            - SPACERAKE_API_SECRET

        '''
        api_url    = os.environ['SPACERAKE_API_URL']
        api_key    = os.environ['SPACERAKE_API_KEY']
        api_secret = os.environ['SPACERAKE_API_SECRET']

        return SpacerakeSDK(api_url, api_key, api_secret)


    def to_sync(self) -> SpacerakeSDKSync:
        '''
        Get a synchronous wrapper around this API.

        Note that this destroys your editor's ability to infer typing and other
        info. The asynchronous version is preferred.
        '''
        return SpacerakeSDKSync(self)


    async def __request_make(self, method, path, data=None, params=None):
        auth_header = f'Bearer {self.api_key}/{self.api_secret}'
        async with httpx.AsyncClient() as client:
            resp = await client.request(
                method,
                self.api_url + path,
                headers={
                    'Authorization': auth_header
                },
                data=data,
                params=params
            )

            return resp


    async def __request_get(self, path, data=None, params=None):
        return await self.__request_make(
            'GET',
            path,
            data=data,
            params=params
        )


    async def __request_post(self, path, data=None, params=None):
        return await self.__request_make(
            'POST',
            path,
            data=data,
            params=params
        )


    async def __abstract_get(self, path, rtype, params=None) -> Any:
        '''
        Automatically managed GET request
        '''
        resp = await self.__request_get(path, params=params)
        resp = APIResponse[rtype].parse_obj(resp.json())

        if resp.error:
            raise SpacerakeAPIException(resp.error)

        return resp.data


    async def __abstract_post(self, path, rtype, params=None, data=None) -> Any:
        '''
        Automatically managed POST request
        '''
        assert data is not None
        resp = await self.__request_post(path, data=data.json(), params=params)
        resp = APIResponse[rtype].parse_obj(resp.json())

        if resp.error:
            raise SpacerakeAPIException(resp.error)

        return resp.data



    async def self_test(self) -> bool:
        '''
        Check that the API connection is valid.

        :return: True if credentials are valid.
        :rtype: bool
        '''
        resp = await self.__request_get('/api/v0/special/auth/test')

        if resp.json()['error'] is None:
            return True

        return False


    async def self_profile(self) -> UserProfile:
        '''
        Gets the current user's profile.

        :return: User profile 
        :rtype: UserProfile
        '''

        return await self.__abstract_get(
            '/api/v0/special/auth/me',
            List[Order]
        )

    
    # --- ORDERS ---

    async def order_list(self) -> List[Order]:
        return await self.__abstract_get(
            '/api/v0/orders/list',
            List[Order]
        )


    async def order_get(self, order_id: str) -> Order:
        return await self.__abstract_get(
            f'/api/v0/orders/order/{order_id}',
            Order
        )


    async def order_get_details(self, order_id: str) -> OrderWithDetails:
        return await self.__abstract_get(
            f'/api/v0/orders/order/{order_id}/details',
            OrderWithDetails
        )



    async def order_get_lineage(self, order_id: str) -> LineageReport:
        return await self.__abstract_get(
            f'/api/v0/orders/order/{order_id}/details',
            LineageReport
        )


    async def order_create(self, order: OrderCreationRequest) -> Order:
        return await self.__abstract_post(
            f'/api/v0/orders/create',
            Order,
            data=order
        )

    # --- SEARCH ---

    async def search(self, search: SearchRequest) -> SearchResponse:
        return await self.__abstract_post(
            f'/api/v0/search/images/polygon',
            SearchResponse,
            data=search
        )


    async def search_get_replay(self, search_id: str) -> SearchResponse:
        return await self.__abstract_get(
            f'/api/v0/search/history/{search_id}',
            SearchResponse
        )


    async def search_list(self) -> List[Search]:
        return await self.__abstract_get(
            f'/api/v0/search/history/list',
            List[Search]
        )

    
    # --- ASSETS ---
    
    async def asset_get(self, asset_id: str) -> Asset:
        return await self.__abstract_get(
            f'/api/v0/assets/asset/{asset_id}',
            Asset
        )


    async def asset_list(self) -> List[Asset]:
        return await self.__abstract_get(
            f'/api/v0/assets/list',
            List[Asset]
        )


    async def asset_get_download_link(self, asset_id: str) -> str:
        resp = await self.__request_get(
            f'/api/v0/assets/asset/{asset_id}/download'
        )

        if resp.status_code == 404:
            raise SpacerakeAPIException(APIResponse[Any].parse_obj(resp.json()).error)

        return resp.headers['location']
        

    async def asset_download_bytes(self, asset_id: str) -> bytes:
        link = await self.asset_get_download_link(asset_id)

        async with httpx.AsyncClient() as client:
            resp = await client.get(link, follow_redirects=True)

            return resp.content


    async def asset_download_file(self, asset_id: str, filename: str):
        content = await self.asset_download_bytes(asset_id)
        with open(filename, 'wb') as fp:
            fp.write(content)



    # --- SAVED LOCATIONS ---

    async def savedlocations_list(self) -> List[SavedLocation]:
        return await self.__abstract_get(
            '/api/v0/savedlocations/list',
            List[SavedLocation]
        )


    async def savedlocations_get(self, location_id: str) -> SavedLocation:
        return await self.__abstract_get(
            f'/api/v0/savedlocations/savedlocation/{location_id}',
            SavedLocation
        )


    async def savedlocations_create(self, location_req: SavedLocationCreationRequest) -> SavedLocation:
        return await self.__abstract_post(
            f'/api/v0/savedlocations/create',
            SavedLocation,
            data=location_req
        )

    
    # --- IMAGES ---

    async def image_get(self, image_id: str) -> Image:
        return await self.__abstract_get(
            f'/api/v0/images/image/{image_id}',
            Image
        )

    async def image_get_thumbnail_link(self, image_id: str) -> str:
        resp = await self.__request_get(
            f'/api/v0/images/image/{image_id}/thumbnail',
        )

        if resp.status_code == 404:
            raise SpacerakeAPIException(APIResponse[Any].parse_obj(resp.json()).error)

        return resp.headers['location']
