import functools
import inspect
import os
from pathlib import Path
from typing import Dict, Optional, TypeVar, List

from git import Repo
from appdirs import user_data_dir
from pydantic import BaseModel, Field
from pydantic.types import SecretStr
from tomli import load as toml_load

from . import console
from .constants import (
    APP_AUTHOR,
    APP_NAME,
    DEFAULT_GLOBAL_CONFIG
)

# config models

class GitlabConfig(BaseModel):
    url:   str
    token: str


class AWSConfig(BaseModel):
    access_token: str
    access_secret: SecretStr
    default_region: str


class GooseGlobalSelfConfig(BaseModel):
    hosted_zone_name: str


class GooseGlobalConfig(BaseModel):
    gitlab: GitlabConfig
    aws: AWSConfig
    goose: GooseGlobalSelfConfig


class RepoConfig(BaseModel):
    repository: str
    name: str
    extra_files: List[str]


TaskSpec   = List[str] | str
TaskConfig = Dict[ str, TaskSpec ]


class HostConfig(BaseModel):
    remote: str
    user: str
    tasks: TaskConfig = Field(default={})


class GooseLocalConfig(BaseModel):
    repo: RepoConfig
    hosts: Dict[str, HostConfig]
    tasks: Optional[TaskConfig] = Field(default={})


# loader convenience class

class ConfigLoader:
    def __init__(self):
        pass

    @staticmethod
    def get_global_config_path() -> Path:
        return Path(user_data_dir(APP_NAME, APP_AUTHOR)).joinpath('config.toml')


    @staticmethod
    def get_local_config_path() -> Optional[Path]:
        def has_config(pth: Path) -> bool:
            pth = pth.joinpath('goose.toml')
            return pth.exists()

        path = Path(os.getcwd())
        
        while True:
            if has_config(path):
                return path.joinpath('goose.toml')

            path = path.parent
            if path == path.parent:
                return None
    
    @staticmethod
    def get_local_project_root() -> Optional[Path]:
        path = ConfigLoader.get_global_config_path()
        if path is None:
            return None

        return path.parents[0]

    @staticmethod
    def create_global_config():
        global_pth = ConfigLoader.get_global_config_path()

        global_pth.parent.mkdir(parents=True, exist_ok=True)
        global_pth.write_text(DEFAULT_GLOBAL_CONFIG)

    @staticmethod
    def load_global_config() -> GooseGlobalConfig:
        global_pth = ConfigLoader.get_global_config_path()

        if not global_pth.exists():
            console.log(f'[yellow]creating default config at {global_pth}[/]')
            ConfigLoader.create_global_config()

        with open(global_pth, 'rb') as fp:
            obj = toml_load(fp)
            return GooseGlobalConfig.parse_obj(obj)


    @staticmethod
    def load_local_config() -> Optional[GooseLocalConfig]:
        local_pth = ConfigLoader.get_local_config_path()
        
        if local_pth is None:
            return None

        with open(local_pth, 'rb') as fp:
            obj = toml_load(fp)
            return GooseLocalConfig.parse_obj(obj)


def get_local_host_by_name(local_config: GooseLocalConfig, hostname: str) -> Optional[HostConfig]:
    for name, host in local_config.hosts.items():
        if name == hostname:
            return host

    return None

def get_local_config_git_repo(local_config_path: Path) -> Optional[Repo]:
    git = local_config_path.parent.joinpath('.git')
    if not git.exists():
        return None

    return Repo(git)
    



def attach_configs(func):

    def attach_config_kwargs(*args, **kwargs):
        # convert the tuple list to a modifiable list
        args = [*args]

        # heavy wizardry to inspect the signature to find out which ones have
        # been assigned
        sig = inspect.signature(func)
        for param_name in sig.parameters:
            # pprint(sig.parameters[param_name].default)
            if sig.parameters[param_name].default is GooseLocalConfig:
                kwargs[param_name] = ConfigLoader.load_local_config()
            if sig.parameters[param_name].default is GooseGlobalConfig:
                kwargs[param_name] = ConfigLoader.load_global_config()
                
        return func(*args, **kwargs)

    # forward docstring
    attach_configs.__doc__ = func.__doc__

    # set the annotations for the function so typing kinda works
    sig = inspect.signature(func)
    for param_name in sig.parameters:
        if sig.parameters[param_name].default is GooseLocalConfig:
            func.__annotations__[param_name] = GooseLocalConfig
        if sig.parameters[param_name].default is GooseGlobalConfig:
            func.__annotations__[param_name] = GooseGlobalConfig

    # pprint(inspect.get_annotations(func))

    # modify signature
    return attach_config_kwargs
