import os
import re
import subprocess
import sys
import time
from typing import Tuple, Optional

import asyncclick as click
from git.remote import Remote
from git.repo.base import Repo
from pydantic import ValidationError
from rich.tree import Tree
from rich.prompt import Confirm, Prompt
from rich.table import Column, Table
from fabric import Connection
from invoke.exceptions import UnexpectedExit, Failure

from . import console, pprint
from .aws import (
    apply_dns_changes,
    create_boto3_session,
    create_elastic_ip,
    get_machine_power_state,
    get_machine_with_name,
    get_zone_id_with_name,
    list_apex_records,
    list_ec2_and_eips,
    set_machine_power_state,
    tag_ec2_instance,
    untag_ec2_instance,
)
from .config import ConfigLoader, GooseLocalConfig, HostConfig, get_local_config_git_repo, get_local_host_by_name


@click.group()
def goose_cli():
    # entrypoint for project-level operations
    pass


@click.group()
def flock_cli():
    # entrypoint for flock-level ops
    pass



def show_config(validate: bool):
    global_config = ConfigLoader.get_global_config_path()
    local_config  = ConfigLoader.get_local_config_path()

    console.print(f'Global configuration path:\n[blue]{global_config}[/]\n')

    if local_config is None:
        console.print('[yellow]No [/][blue]goose.toml[/][yellow] configuration file found in this directory or it\'s parents.')
    else:
        console.print(f'Project configuration path:\n[blue]{local_config}[/]')

    if not validate:
        return

    try:
        console.print('\nGlobal configuration:')
        pprint(ConfigLoader.load_global_config())
    except ValidationError as e:
        console.print('[red]Global configuration did not load successfully!\n')
        console.print(e)

    if local_config is None:
        return

    try:
        console.print('\nLocal configuration:')
        pprint(ConfigLoader.load_local_config())
    except ValidationError as e:
        console.print('[red]Local configuration did not load successfully!\n')
        console.print(e)


@flock_cli.command('config', help='Show active configuration paths.')
@click.option('-v', '--validate', help='Validate config files.', is_flag=True)
def flock_config_paths(validate):
    show_config(validate)


@goose_cli.command('config', help='Show active configuration paths.')
@click.option('-v', '--validate', help='Validate config files.', is_flag=True)
def goose_config_paths(validate):
    show_config(validate)


@flock_cli.command('login', help='Set up the goose global configuration.')
def login():
    ConfigLoader.load_global_config()
    gc_path = ConfigLoader.get_global_config_path()
    editor = os.getenv('EDITOR')

    if editor is None:
        console.print('[red]no editor is configured![/]')
        console.print('Please use your preferred text editor to edit:\n')
        console.print(f'[blue]{gc_path}[/]')
        return


    subprocess.call([editor, gc_path])


@flock_cli.command('list', help='List all machines in the AWS account.')
@click.option('-p', '--project', help='Filter by project')
@click.option('-r', '--running', help='Only show running instances', is_flag=True)
@click.option('-i', '--id', help='Show the ID of the instance', is_flag=True)
@click.option('-d', '--domain', help='Show the URL of the instance', is_flag=True)
def list_machines(project, running, id, domain):
    config = ConfigLoader.load_global_config()
    sesh = create_boto3_session(config)

    with console.status('Getting EC2 instance list...'):
        hosts_and_ips = list_ec2_and_eips(sesh)

    with console.status('Getting hosted zone...'):
        zone_id = get_zone_id_with_name(config.goose.hosted_zone_name, sesh)
        assert zone_id is not None
        all_records = list_apex_records(zone_id, sesh)
    
    # create a record map for easy searching later
    record_map = { r['ResourceRecords'][0]['Value']: r for r in all_records } 

    # hack to kick the unassigned projects down to the bottom
    hosts_and_ips.sort(key=lambda h: h['project'] if h['project'] else 'zzzzzzz')

    # apply filters
    if project is not None:
        hosts_and_ips = [h for h in hosts_and_ips if h['project'] and h['project'].lower() == project.lower()]
    if running:
        hosts_and_ips = [h for h in hosts_and_ips if h['running'] == 'running']


    # create the table for outputting
    table = Table(
        Column(header='Name', style='bold blue'),
        Column(header='Hostname', style='blue'),
        'Project',
        'Type',
        # expand=True
    )
    if id:
        table.add_column(header='Instance ID', style='green')
    if domain:
        table.add_column(header='Domain name', style='blue')

    table.add_column(header='DNS?', max_width=4)
    table.add_column(header='EIP?', max_width=4)
    table.add_column(header='On?', max_width=4)

    for _, host in enumerate(hosts_and_ips):
        # determine whether we have an A record associated with the Elastic IP
        # on this instance
        dns_status = '[red]:heavy_multiplication_x:[/]'
        domain_name = None
        if host['associated_ip'] in record_map:
            dns_status = '[green]:heavy_check_mark:[/]'
            domain_name = record_map[host['associated_ip']]['Name']

        # check if it's running
        run_status = '[red]:heavy_multiplication_x:[/]'
        if host['running'] == 'running':
            run_status = '[green]:heavy_check_mark:[/]'

        # check if it *has* an elastic ip
        eip_status = '[red]:heavy_multiplication_x:[/]'
        if host['associated_ip'] is not None:
            eip_status = '[green]:heavy_check_mark:[/]'

        row_data = [
            host['name'],
            host['hostname'],
            host['project'],
            host['instance_type']
        ]

        if id:
            row_data.append(host['instance_id'])
        if domain:
            row_data.append(domain_name)

        row_data += [
            dns_status,
            eip_status,
            run_status
        ]

        table.add_row(*row_data)

        # if i+1 != len(hosts_and_ips) and hosts_and_ips[i+1]['project'] != host['project']:
        #     table.add_section()

    console.print(table)

    # pprint(record_map)


@flock_cli.command('hostname', help='Set a hostname for a machine by name.')
@click.option('-c', '--clear', is_flag=True, help='Clear a hostname')
@click.option('-o', '--overwrite', is_flag=True, help='Override an existing hostname without asking')
@click.argument('machine_name')
def machine_set_hostname(machine_name, clear, overwrite):
    config = ConfigLoader.load_global_config()
    sesh = create_boto3_session(config)

    # get the host from the list of hosts
    with console.status('Getting EC2 instance list...'):
        host = get_machine_with_name(machine_name, sesh)

    if host is None:
        console.print(f'[red]No host matching {machine_name} found![/]')
        return

    # check to see if, in the case that we already have a hostname, the user
    # really does want to overwrite it
    if host['hostname'] is not None:
        console.print(f'[yellow]Host "{machine_name}" already has a hostname: [blue]{host["hostname"]}[/][yellow]!')
        
        if clear and not Confirm.ask('Delete?'):
            console.print('Aborted!')
            return
        elif not overwrite and not clear and not Confirm.ask('Overwrite?'):
            console.print('Aborted!')
            return

    if clear:
        hostname = host['hostname']
        untag_ec2_instance(
            host['instance_id'],
            'Hostname',
            host['hostname'],
            sesh
        )
        console.print(f'[green]remove hostname {hostname} to instance {machine_name} ({host["instance_id"]}) successfully!')
        return

    # prompt the user for the host name
    while True:
        hostname = Prompt.ask(
            'Please select a new hostname'
        )

        if not re.match(r'^[a-zA-Z0-9\.\-\_]+$', hostname):
            console.print('[yellow]hostname must only contain a-z, A-Z, 0-9, ., -, _')
        else:
            break

    tag_ec2_instance(
        host['instance_id'],
        'Hostname',
        hostname,
        sesh
    )
    
    console.print(f'[green]applied hostname {hostname} to instance {machine_name} ({host["instance_id"]}) successfully!')


@flock_cli.command('power', help='View and set machine power states')
@click.argument('machine_name')
@click.option('-s', '--state', 'requested_state', help='state to set to', type=click.Choice(['start', 'stop']))
@click.option('-w', '--wait', help='wait for the machine power state', is_flag=True)
def machine_power(machine_name, requested_state, wait):
    config = ConfigLoader.load_global_config()
    sesh = create_boto3_session(config)

    # get the host from the list of hosts
    with console.status('Getting EC2 instance list...'):
        host = get_machine_with_name(machine_name, sesh)

    if host is None:
        console.print(f'[red]No host matching {machine_name} found![/]')
        return

    instance_id = host['instance_id']

    # power state
    power_state = get_machine_power_state(instance_id, sesh)

    # power state
    console.print(f'Machine power state: [green]{power_state}[/]')

    if requested_state is None:
        # no change is required
        # our job here is done
        return

    # make the request
    with console.status(f'Setting power state to [green]{requested_state}[/]...'):
        set_machine_power_state(instance_id, requested_state, sesh)


    # if we're not supposed to wait, return early
    if not wait:
        return


    start_time = time.time()
    with console.status(f'Polling power state...'):
        while True:
            power_state = get_machine_power_state(instance_id, sesh)

            elapsed = int(time.time() - start_time)

            # this is a code smell
            if requested_state == 'start' and power_state == 'running' or \
                    requested_state == 'stop' and power_state == 'stopped':
                console.print(f'desired state [green]{power_state}[/] reached after [blue]{elapsed} sec[/].') 
                return
            else:
                console.print(f'polled power state [green]{power_state}[/], [blue]{elapsed} sec[/] elapsed.')

            if elapsed > 180:
                console.print('[red]Stopping polling after 3 minutes![/]')
                sys.exit(1)
                

            time.sleep(1)


@flock_cli.command('dns-sync', help='Sync the dns names with the hostname tags')
@click.option('-d', '--dry-run', is_flag=True, help='Only calculate the change plan without executing it')
@click.option('--ttl', default=600, help='Set the TTL for new DNS records')
def dns_sync(dry_run, ttl):
    config = ConfigLoader.load_global_config()
    sesh = create_boto3_session(config)

    with console.status('Getting EC2 instance list...'):
        hosts_and_ips = list_ec2_and_eips(sesh)

    with console.status('Getting hosted zone...'):
        zone_id = get_zone_id_with_name(config.goose.hosted_zone_name, sesh)
        assert zone_id is not None
        all_records = list_apex_records(zone_id, sesh)
    
    # create a record map for easy searching later
    record_map = { r['ResourceRecords'][0]['Value']: r for r in all_records } 

    # pprint(hosts_and_ips)
    # pprint(record_map)

    # calculate the actions required

    # first, run through all the hosts and check if they need an EIP or an A record
    plan_tree = Tree('[bold]Action plan')
    plan_tree_instances = plan_tree.add('[green]Instance info')
    instance_actions = []
    assigned_a_records = []
    for host in hosts_and_ips:
        # if there's no hostname, we don't need to touch anything
        if host['hostname'] is None:
            continue

        # for posterity
        plan_tree_node = plan_tree_instances.add(f'[blue]{host["name"]}')
        plan_tree_node.add(f'hostname: [blue]{host["hostname"]}')

        # first, check if we need to assign an elastic IP to this instance
        needs_eip = (host['associated_ip'] is None)
        plan_tree_node.add(f'elastic ip: [blue]{host["associated_ip"]}')

        # next, calculate our  domain name
        requested_record_name = f'{host["hostname"]}.{config.goose.hosted_zone_name}.'
        plan_tree_node.add(f'requested record: [blue]{requested_record_name}')

        # next, see if we need to do anything to update our DNS record
        needs_dns_record_sync = False
        if needs_eip:
            # if we don't have an elastic IP, we need to create a record
            needs_dns_record_sync = True
        else:
            if not host['associated_ip'] in record_map:
                needs_dns_record_sync = True
            elif record_map[host['associated_ip']]['Name'] != requested_record_name:
                needs_dns_record_sync = True

        if not needs_dns_record_sync:
            assigned_a_records.append(requested_record_name)
        

        action_tree = plan_tree_node.add('[orange]actions')
        action_tree.add(f'needs EIP: [blue]{needs_eip}')
        action_tree.add(f'needs domain sync: [blue]{needs_dns_record_sync}')

        # create the action
        action = {
            'instance_id': host['instance_id'],
            'name': host['name'],
            'ip': host['associated_ip'],
            'host': host,
            'requested_record_name': requested_record_name,
            'needs_eip': needs_eip,
            'needs_dns_record_sync': needs_dns_record_sync
        }


        instance_actions.append(action)

    # next, report which EIPs we are creating
    total_action_count = 0
    plan_tree_eips = plan_tree.add('[green]Elastic IP operations').add('Creations')
    for action in instance_actions:
        if not action['needs_eip']:
            continue

        total_action_count += 1
        plan_tree_eips.add(f'[blue]{action["name"]}')

    # finally, figure out which A records we don't need anymore
    plan_tree_a = plan_tree.add('[green]A record operations')
    plan_tree_a_new = plan_tree_a.add('Creations')
    plan_tree_a_del = plan_tree_a.add('[red]Deletions')
    records_to_purge = []
    for record in all_records:
        if not record['Name'] in assigned_a_records:
            records_to_purge.append(record)
            total_action_count += 1
            plan_tree_a_del.add('[yellow]' + record['Name'])

    # for posterity, write down which domains go where
    for action in instance_actions:
        if not action['needs_dns_record_sync']:
            continue

        total_action_count += 1

        rrn = action['requested_record_name']
        ip  = action['ip']
        
        if action['needs_eip']:
            plan_tree_a_new.add(f'[yellow]{rrn}[/] -> [blue]new EIP for {action["name"]}')
        else:
            plan_tree_a_new.add(f'[yellow]{rrn}[/] -> {ip}')


    console.print(plan_tree)

    if dry_run:
        return

    if total_action_count == 0:
        console.print('\n[green]No action to take.[/]')
        return

    if not Confirm.ask(f'Apply this plan ({total_action_count} actions)?'):
        console.print('No action taken.')
        return

    # the magic happens here
    
    # first, create EIPs where necessary
    eip_action_count = 0
    for action in instance_actions:
        if not action['needs_eip']:
            continue

        with console.status(f'Creating EIP for [blue]{action["name"]}[/]...'):

            eip_alloc = create_elastic_ip(
                action['instance_id'],
                action['name'],
                sesh
            )

            console.print(f'Associated EIP [green]{eip_alloc["PublicIp"]}[/] for [blue]{action["name"]}[/].')

            action['ip'] = eip_alloc['PublicIp']
            eip_action_count += 1

    # pprint(instance_actions)

    # next, calculate the A record changes
    # first, additions...
    a_record_creations = []
    for action in instance_actions:
        if not action['needs_dns_record_sync']:
            continue

        a_record_creations.append({
            'Action': 'UPSERT',
            'ResourceRecordSet': {
                'Name': action['requested_record_name'],
                'Type': 'A',
                'TTL': ttl,
                'ResourceRecords': [{'Value': action['ip']}]
            }
        })

    # ... and deletions
    a_record_deletions = []
    for record in records_to_purge:
        a_record_deletions.append({
            'Action': 'DELETE',
            'ResourceRecordSet': record
        })


    dns_changes = len(a_record_creations) + len(a_record_deletions)
    if dns_changes > 0:
        with console.status(f'Applying {dns_changes} DNS changes...'):
            if len(a_record_deletions) != 0:
                apply_dns_changes(
                    zone_id,
                    a_record_deletions,
                    'goose DNS sync (deletion step)',
                    sesh
                )

            if len(a_record_creations) != 0:
                apply_dns_changes(
                    zone_id,
                    a_record_creations,
                    'goose DNS sync (creation step)',
                    sesh
                )

    console.print(f'[green]Applied {eip_action_count} instance actions and {dns_changes} DNS changes!')


# @cli.group('project', help='Project management commands.')
# def group_project():
#     pass

def get_host_connection(local_config: GooseLocalConfig, hostname: str):
    host = get_local_host_by_name(local_config, hostname)
    if host is None:
        console.print(f'[yellow]No such host "{hostname}"!')
        raise RuntimeError('invalid host')

    return Connection(
        host.remote,
        user=host.user
    )


def git_operations_setup(local_config: GooseLocalConfig, hostname: str) -> Tuple[Repo, HostConfig, Optional[Remote]]:
    local_path = ConfigLoader.get_local_config_path()
    assert local_path is not None

    host = get_local_host_by_name(local_config, hostname)

    if host is None:
        console.print(f'[yellow]No such host "{hostname}"!')
        raise RuntimeError('invalid host')

    repo = get_local_config_git_repo(local_path)

    if repo is None:
        console.print(f'[yellow]No git repository found!')
        raise RuntimeError('no git repo')

    expected_remote_url = f'{host.user}@{host.remote}:{local_config.repo.name}'

    host_remote = None
    for remote in repo.remotes:
        if remote.url == expected_remote_url:
            host_remote = remote
    return (repo, host, host_remote)


@goose_cli.command('connect', help='Connect to an instance.')
@click.argument('host')
def project_connect(host):
    lconfig = ConfigLoader.load_local_config()
    assert lconfig is not None

    hostname = host
    host = get_local_host_by_name(lconfig, hostname)
    if host is None:
        console.print(f'[yellow]No such host "{hostname}"!')
        return

    conn = Connection(
        host.remote,
        user=host.user
    )

    try:
        conn.shell()
    except UnexpectedExit as e:
        pass



@goose_cli.command('install-remote', help='Add a git remote for the host on the local repo.')
@click.argument('host')
@click.option('-n', '--remote-name', help='What to call the git remote')
def project_install_remote(host, remote_name):
    lconfig = ConfigLoader.load_local_config()
    assert lconfig is not None

    hostname = host
    try:
        repo, host, _ = git_operations_setup(lconfig, host)
        conn = get_host_connection(lconfig, hostname)
    except:
        return

    # host = get_local_host_by_name(lconfig, hostname)
    # if host is None:
    #     console.print(f'[yellow]No such host "{hostname}"!')
    #     return
    
    # repo = get_local_config_git_repo(local_path)
    # if repo is None:
    #     console.print(f'[yellow]No git repository found!')
    #     return

    requested_remote_url = f'{host.user}@{host.remote}:{lconfig.repo.name}'
    requested_remote_name = f'deploy-{hostname}' if remote_name is None else remote_name

    # print(repo)
    # print(requested_remote_url)
    # print(requested_remote_name)

    # check to see if the remote has already been added
    for remote in repo.remotes:
        if remote.url == requested_remote_url:
            console.print(f'[red]git remote {remote.name} ({remote.url}) already exists!')
            return

    # repo.remotes.origin.url
    remote = repo.create_remote(
        requested_remote_name,
        requested_remote_url
    )

    with console.status('creating remote repo...'):
        conn.run(f'which git || sudo yum install -y git', pty=True)
        conn.run(f'mkdir -p {lconfig.repo.name} && cd {lconfig.repo.name} && git init --initial-branch={repo.active_branch}', pty=True)
        conn.run(f'git config --global receive.denyCurrentBranch updateInstead')

    console.print(f'[green]Created remote {remote.name} ({remote.url}) on local repo successfully!')


@goose_cli.command('push', help='Push this repo to the remote.')
@click.argument('host')
@click.option('-f', '--force', help='Force push', is_flag=True)
@click.option('-i', '--ignore-errors', help='Ignore errors.', is_flag=True)
@click.option('-e', '--only-extras', help='Skip git push and go right to the extra file transfer step.', is_flag=True)
@click.option('-r', '--remote-stash', help='Stash changes on the remote repo.', is_flag=True)
def project_push_remote(host, force, ignore_errors, only_extras, remote_stash):
    lconfig = ConfigLoader.load_local_config()
    assert lconfig is not None

    project_root = ConfigLoader.get_local_project_root()
    assert project_root is not None # should always be true if we have a local config

    hostname = host
    try:
        repo, host, remote = git_operations_setup(lconfig, host)
        conn = get_host_connection(lconfig, hostname)
    except:
        return

    if remote is None:
        console.print(f'[yellow]No git remote for {hostname} found!')
        return

    if remote_stash:
        console.print(f'[blue]Stashing remote changes...')
        conn.run(f'cd {lconfig.repo.name} && git stash')

    if not only_extras:
        console.print(f'[blue]Pushing the git repo...')
        # with console.status(f'Pushing to {remote.url}...'):
        if ignore_errors:
            remote.push(
                repo.active_branch,
                force=force,
                verbose=True,
                allow_unsafe_options=True,
                no_verify=True
            )
        else:
            remote.push(
                repo.active_branch,
                force=force,
                allow_unsafe_options=True,
                verbose=True,
                no_verify=True
            ).raise_if_error()

        conn.run(f'cd {lconfig.repo.name} && git checkout {repo.active_branch}', pty=True)

    with console.status(f'[blue]Uploading extra files...'):
        os.chdir(project_root)
        for fn in lconfig.repo.extra_files:
            if not os.path.exists(fn):
                console.print(f'[red]Could not find local file "{fn}"!')
                continue
            conn.put(fn, os.path.join(lconfig.repo.name, fn))

    console.print(f'[green]Push to {remote.url} completed.')
    

@goose_cli.command('run', help='Run a task on the host.')
@click.argument('host')
@click.argument('task')
@click.option('-i', '--ignore', help='Ignore errors.', is_flag=True)
def run_task(host, task, ignore):
    lconfig = ConfigLoader.load_local_config()
    assert lconfig is not None

    if lconfig.tasks is None:
        console.print(f'[yellow]No tasks specified!')
        return

    try:
        host_spec = get_local_host_by_name(lconfig, host)
    except:
        return
    assert host_spec is not None

    all_tasks = list(lconfig.tasks.items()) + list(host_spec.tasks.items())
    
    selected_task = None
    for task_name, task_spec in all_tasks:
        if task_name == task:
            selected_task = task_spec
            break

    if selected_task is None:
        console.print(f'[yellow]No such task [b red]{task}[/b red] available for [b red]{host}[/b red]!')
        return
    
    if type(selected_task) is str:
        selected_task = [selected_task]

    try:
        conn = get_host_connection(lconfig, host)
    except:
        return

    for idx, command in enumerate(selected_task):
        console.print(f'[blue]:: {idx+1} / {len(selected_task)} :: [green]{command}[/][blue] ::')
        try:
            conn.run(command, pty=True)
        except UnexpectedExit as e:
            if not ignore:
                console.print(f'[yellow]:: Received unexpected return code {e.result.exited}, aborting... ::')
                return
        except Failure as f:
            console.print(f'[yellow]:: Unexpected failure (details below), aborting... ::')
            console.print(f)




